/*
 * Copyright (C) 2004-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: time_widget.c,v 1.6 2005/01/24 19:24:11 dsalt Exp $
 *
 * a nice black time display area
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <xine.h>

#include "time_widget.h"
#include "utils.h"
#include "globals.h"
#include "player.h"

#define TIME_WIDGET_WIDTH (90)
#define TIME_WIDGET_HEIGHT (40)

struct time_widget_s {

  GtkWidget            *w;

  PangoLayout          *pl, *pl_big;
  PangoFontDescription *pfd, *pfd_big;

  GdkPixmap	       *pixmap;
  gint			width, height;
  GdkGC                *black_gc, *blue_gc, *white_gc;
};

static void paint_tw (time_widget_t *tw)
{
  gint pos_stream, pos_time, length_time;

  if (!tw->pixmap)
    return;

  gdk_draw_rectangle (tw->pixmap, tw->black_gc, TRUE, 0, 0,
		      tw->width, tw->height);

  if ((xine_get_status (stream) == XINE_STATUS_PLAY) &&
      (xine_get_pos_length (stream, &pos_stream, &pos_time, &length_time)))
  {
    gchar curtime_str[64], length_str[64];

    int_to_timestring (pos_time / 1000, curtime_str, sizeof (curtime_str));
    if (length_time > 0)
      int_to_timestring (length_time / 1000, length_str, sizeof (length_str));
    else
      strncpy (length_str, pos_stream <= 0 ? _("live") : _("??:??:??"),
	       sizeof (length_str));

    pango_layout_set_text (tw->pl_big, curtime_str, -1);
    gdk_draw_layout (tw->pixmap, tw->blue_gc, 7, 2, tw->pl_big);

    pango_layout_set_text (tw->pl, length_str, -1);
    gdk_draw_layout (tw->pixmap, tw->blue_gc, 7, 23, tw->pl);
  }
}

static void alloc_color(GtkWidget *window, GdkGC *gc, gint r, gint g, gint b) {

  GdkColor *color=malloc(sizeof(GdkColor));

  if (gc) {
    color->red = r; color->green = g; color->blue = b;
    if (!gdk_colormap_alloc_color (gtk_widget_get_colormap (window), color, FALSE, TRUE)) {
      printf ("failed to alloc color (%d/%d/%d)\n",color->red,color->green,color->blue);
      exit(1);
    }
    gdk_gc_set_foreground (gc, color);
  }
}

static gint update_time_widget_cb (gpointer tw)
{
  gdk_threads_enter ();
  paint_tw (tw);
  gtk_widget_queue_draw (((time_widget_t *)tw)->w);
  gdk_threads_leave ();
  return TRUE;
}

static void tw_create_pixmap (time_widget_t *tw, gint width, gint height)
{
  GtkWidget *window = gtk_widget_get_toplevel (tw->w);

  tw->width = width;
  tw->height = height;
  tw->pixmap = gdk_pixmap_new (tw->w->window, width, height, -1);

  tw->black_gc = gdk_gc_new (tw->pixmap);
  alloc_color (window, tw->black_gc, 0, 0, 0);

  tw->white_gc = gdk_gc_new (tw->pixmap);
  alloc_color (window, tw->white_gc, 65535, 65535, 65535);

  tw->blue_gc = gdk_gc_new (tw->pixmap);
  alloc_color (window, tw->blue_gc, 20000, 20000, 65535);

  gdk_draw_rectangle (tw->pixmap, tw->black_gc, TRUE, 0, 0, width, height);
  gtk_image_set_from_pixmap (GTK_IMAGE(tw->w), tw->pixmap, NULL);
}

static gboolean tw_configure_cb (GtkWidget *widget, GtkAllocation *event,
				 time_widget_t *tw)
{
  if (tw->w->parent && tw->w->window &&
      (!tw->pixmap || event->width != tw->width || event->height != tw->height))
    tw_create_pixmap (tw, event->width, event->height);
  return FALSE;
}

time_widget_t *create_time_widget (GtkWidget **w)
{
  time_widget_t *tw = (time_widget_t *) malloc (sizeof (time_widget_t));
  memset (tw, 0, sizeof (*tw));
  *w = tw->w = gtk_image_new ();
  gtk_widget_set_size_request (tw->w, TIME_WIDGET_WIDTH, TIME_WIDGET_HEIGHT);
  g_signal_connect (G_OBJECT(tw->w), "size-allocate",
		    G_CALLBACK(tw_configure_cb), tw);
  return tw;
}

void postinit_time_widget (time_widget_t *tw)
{
  GtkRequisition size;
  gtk_widget_size_request (tw->w, &size);

  /* pango stuff (antialiased font rendering) */

  tw->pfd = pango_font_description_from_string ("Sans Bold 8");
  tw->pl = gtk_widget_create_pango_layout (tw->w, NULL);
  pango_layout_set_font_description (tw->pl, tw->pfd);

  tw->pfd_big = pango_font_description_from_string ("Sans Bold 12");
  tw->pl_big = gtk_widget_create_pango_layout (tw->w, NULL);
  pango_layout_set_font_description (tw->pl_big, tw->pfd_big);

  if (tw->w->parent && tw->w->window && !tw->pixmap)
    tw_create_pixmap (tw, size.width, size.height);

  gtk_timeout_add (500, update_time_widget_cb, tw); 
}
